const express = require('express');
const { executeQuery } = require('../config/db');
const router = express.Router();

// POST API - Create a new blog
router.post('/create', async (req, res) => {
  const { title, heading, paragraph, author, detailed_description, image_base64 } = req.body;
  
  console.log('Blog creation request received:', { title, heading, author });
  
  try {
    // Validate required fields
    if (!title || !heading || !paragraph || !author) {
      return res.status(400).json({ 
        message: 'Missing required fields: title, heading, paragraph, and author are required' 
      });
    }

    const results = await executeQuery(
      'INSERT INTO blogs (title, heading, paragraph, author, detailed_description, image_base64, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())',
      [title, heading, paragraph, author, detailed_description || null, image_base64 || null]
    );
    
    console.log('Blog inserted into database:', results);
    res.status(201).json({ 
      message: 'Blog created successfully',
      blog_id: results.insertId 
    });
  } catch (err) {
    console.error('Blog creation error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// GET API - Get all blogs (without detailed description and image)
router.get('/all', async (req, res) => {
  console.log('Get all blogs request received');
  
  try {
    const results = await executeQuery(
      'SELECT id, title, heading, paragraph, author, created_at FROM blogs ORDER BY created_at DESC'
    );
    
    console.log(`Found ${results.length} blogs`);
    res.json({ 
      blogs: results,
      count: results.length 
    });
  } catch (err) {
    console.error('Get all blogs error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// GET API - Get blog by ID (with detailed description and image)
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  
  console.log('Get blog by ID request received:', id);
  
  try {
    const results = await executeQuery(
      'SELECT id, title, heading, paragraph, author, detailed_description, image_base64, created_at FROM blogs WHERE id = ?',
      [id]
    );
    
    if (results.length === 0) {
      console.log('No blog found for ID:', id);
      return res.status(404).json({ message: 'Blog not found' });
    }
    
    console.log('Blog found for ID:', id);
    res.json(results[0]);
  } catch (err) {
    console.error('Get blog by ID error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// DELETE API - Delete blog by ID
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  
  console.log('Delete blog request received for ID:', id);
  
  try {
    // First check if the blog exists
    const checkResults = await executeQuery(
      'SELECT id FROM blogs WHERE id = ?',
      [id]
    );
    
    if (checkResults.length === 0) {
      console.log('No blog found for deletion with ID:', id);
      return res.status(404).json({ message: 'Blog not found' });
    }
    
    // Delete the blog
    const results = await executeQuery(
      'DELETE FROM blogs WHERE id = ?',
      [id]
    );
    
    console.log('Blog deleted successfully with ID:', id);
    res.json({ 
      message: 'Blog deleted successfully',
      deleted_id: id 
    });
  } catch (err) {
    console.error('Delete blog error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = router;
