// route/returnorders.js
const express = require('express');
const router = express.Router();
const { executeQuery } = require('../config/db');

// GET all
router.get('/', async (req, res) => {
  try {
    const rows = await executeQuery(
      'SELECT id, returnOrderNo, perfume, amountLoss, createdAt, updatedAt FROM return_orders ORDER BY id DESC'
    );
    res.json(rows);
  } catch (e) {
    console.error('Database query error:', e);
    res.status(500).json({ message: 'Failed to fetch return orders', error: e.message });
  }
});

// CREATE one
router.post('/', async (req, res) => {
  try {
    const { returnOrderNo, perfume, amountLoss } = req.body;

    if (!returnOrderNo || !perfume || amountLoss === undefined) {
      return res.status(400).json({ message: 'Missing required fields' });
    }

    const result = await executeQuery(
      'INSERT INTO return_orders (returnOrderNo, perfume, amountLoss) VALUES (?,?,?)',
      [returnOrderNo, perfume, amountLoss]
    );

    const rows = await executeQuery(
      'SELECT id, returnOrderNo, perfume, amountLoss, createdAt, updatedAt FROM return_orders WHERE id = ?',
      [result.insertId]
    );

    res.status(201).json(rows[0]);
  } catch (e) {
    console.error('Database query error:', e);
    res.status(500).json({ message: 'Failed to create return order', error: e.message });
  }
});

// UPDATE one
router.put('/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const { returnOrderNo, perfume, amountLoss } = req.body;

    if (!id) return res.status(400).json({ message: 'Invalid id' });

    const result = await executeQuery(
      'UPDATE return_orders SET returnOrderNo=?, perfume=?, amountLoss=? WHERE id=?',
      [returnOrderNo, perfume, amountLoss, id]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Return order not found' });
    }

    const rows = await executeQuery(
      'SELECT id, returnOrderNo, perfume, amountLoss, createdAt, updatedAt FROM return_orders WHERE id = ?',
      [id]
    );

    res.json(rows[0]);
  } catch (e) {
    console.error('Database query error:', e);
    res.status(500).json({ message: 'Failed to update return order', error: e.message });
  }
});

// DELETE one
router.delete('/:id', async (req, res) => {
  try {
    const id = Number(req.params.id);
    if (!id) return res.status(400).json({ message: 'Invalid id' });

    const result = await executeQuery('DELETE FROM return_orders WHERE id=?', [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Return order not found' });
    }

    res.json({ success: true });
  } catch (e) {
    console.error('Database query error:', e);
    res.status(500).json({ message: 'Failed to delete return order', error: e.message });
  }
});

// BULK insert (expects array of objects in body)
router.post('/bulk', async (req, res) => {
  try {
    const items = Array.isArray(req.body) ? req.body : [];
    if (!items.length) return res.status(400).json({ message: 'No data provided' });

    // Validate rows and build values
    const values = [];
    for (const row of items) {
      const returnOrderNo = row.returnOrderNo ?? row['Return Order No'] ?? row['returnOrderNo'];
      const perfume = row.perfume ?? row['Return Perfume'] ?? row['perfume'];
      const amountLoss = row.amountLoss ?? row['Return Amount Loss'] ?? row['amountLoss'];

      if (!returnOrderNo || !perfume || amountLoss === undefined || amountLoss === null) {
        return res.status(400).json({ message: 'Invalid row detected. Each row needs returnOrderNo, perfume, amountLoss' });
      }

      values.push([String(returnOrderNo), String(perfume), Number(amountLoss)]);
    }

    const result = await executeQuery(
      'INSERT INTO return_orders (returnOrderNo, perfume, amountLoss) VALUES ?',
      [values]
    );

    res.status(201).json({ success: true, inserted: result.affectedRows });
  } catch (e) {
    console.error('Database query error:', e);
    res.status(500).json({ message: 'Failed bulk upload', error: e.message });
  }
});

module.exports = router;
