const express = require('express');
const crypto = require('crypto');
const { executeQuery } = require('../config/db');
const jwt = require('jsonwebtoken');
const axios = require('axios');
const router = express.Router();

// Middleware to verify JWT token (for protected routes)
const authenticate = (req, res, next) => {
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) return res.status(401).json({ message: 'No token provided' });

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (err) {
    res.status(401).json({ message: 'Invalid token' });
  }
};

// Middleware to verify Shopify webhook
const verifyShopifyWebhook = (req, res, next) => {
  const hmac = req.get('X-Shopify-Hmac-Sha256');
  const body = JSON.stringify(req.body);
  const hash = crypto
    .createHmac('sha256', process.env.SHOPIFY_WEBHOOK_SECRET)
    .update(body, 'utf8')
    .digest('base64');

  if (hash !== hmac) {
    return res.status(401).send('Unauthorized');
  }

  next();
};

// Webhook endpoint for Shopify order fulfillment
router.post('/webhook/orders/fulfilled', express.raw({type: 'application/json'}), async (req, res) => {
  try {
    // Parse the webhook payload (should be raw buffer now)
    const order = JSON.parse(req.body.toString());
    
    // Verify webhook authenticity
    const hmac = req.get('X-Shopify-Hmac-Sha256');
    const hash = crypto
      .createHmac('sha256', process.env.SHOPIFY_WEBHOOK_SECRET)
      .update(req.body, 'utf8')
      .digest('base64');

    if (hash !== hmac) {
      return res.status(401).send('Unauthorized');
    }

    // Extract order data
    const orderData = {
      shopify_order_id: order.id,
      order_number: order.name || order.order_number,
      email: order.email,
      customer_name: order.customer ? `${order.customer.first_name || ''} ${order.customer.last_name || ''}`.trim() : '',
      customer_phone: order.customer ? order.customer.phone : null,
      total_price: parseFloat(order.total_price),
      currency: order.currency,
      financial_status: order.financial_status,
      fulfillment_status: order.fulfillment_status,
      order_status_url: order.order_status_url,
      line_items: JSON.stringify(order.line_items),
      shipping_address: JSON.stringify(order.shipping_address),
      billing_address: JSON.stringify(order.billing_address),
      shopify_created_at: new Date(order.created_at),
      shopify_updated_at: new Date(order.updated_at),
      tags: order.tags,
      note: order.note,
      processed_at: order.processed_at ? new Date(order.processed_at) : null,
      gateway: order.gateway,
      test: order.test || false,
      raw_data: JSON.stringify(order)
    };

    // Insert or update order in database
    const insertQuery = `
      INSERT INTO shopify_orders (
        shopify_order_id, order_number, email, customer_name, customer_phone,
        total_price, currency, financial_status, fulfillment_status, order_status_url,
        line_items, shipping_address, billing_address, shopify_created_at, shopify_updated_at,
        tags, note, processed_at, gateway, test, raw_data
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        order_number = VALUES(order_number),
        email = VALUES(email),
        customer_name = VALUES(customer_name),
        customer_phone = VALUES(customer_phone),
        total_price = VALUES(total_price),
        currency = VALUES(currency),
        financial_status = VALUES(financial_status),
        fulfillment_status = VALUES(fulfillment_status),
        order_status_url = VALUES(order_status_url),
        line_items = VALUES(line_items),
        shipping_address = VALUES(shipping_address),
        billing_address = VALUES(billing_address),
        shopify_updated_at = VALUES(shopify_updated_at),
        tags = VALUES(tags),
        note = VALUES(note),
        processed_at = VALUES(processed_at),
        gateway = VALUES(gateway),
        test = VALUES(test),
        raw_data = VALUES(raw_data),
        updated_at = CURRENT_TIMESTAMP
    `;

    const values = [
      orderData.shopify_order_id,
      orderData.order_number,
      orderData.email,
      orderData.customer_name,
      orderData.customer_phone,
      orderData.total_price,
      orderData.currency,
      orderData.financial_status,
      orderData.fulfillment_status,
      orderData.order_status_url,
      orderData.line_items,
      orderData.shipping_address,
      orderData.billing_address,
      orderData.shopify_created_at,
      orderData.shopify_updated_at,
      orderData.tags,
      orderData.note,
      orderData.processed_at,
      orderData.gateway,
      orderData.test,
      orderData.raw_data
    ];

    await executeQuery(insertQuery, values);
    
    console.log(`Shopify order ${orderData.order_number} processed successfully`);
    res.status(200).json({ message: 'Order processed successfully' });

  } catch (error) {
    console.error('Webhook processing error:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
});

// Webhook endpoint for order creation (optional)
router.post('/webhook/orders/create', express.raw({type: 'application/json'}), (req, res) => {
  try {
    const order = JSON.parse(req.body.toString());
    
    // Verify webhook authenticity
    const hmac = req.get('X-Shopify-Hmac-Sha256');
    const hash = crypto
      .createHmac('sha256', process.env.SHOPIFY_WEBHOOK_SECRET)
      .update(req.body, 'utf8')
      .digest('base64');

    if (hash !== hmac) {
      return res.status(401).send('Unauthorized');
    }

    console.log(`New Shopify order created: ${order.name || order.order_number}`);
    res.status(200).json({ message: 'Order creation webhook received' });
  } catch (error) {
    console.error('Order creation webhook error:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
});

// Get all Shopify orders (protected route)
router.get('/orders', authenticate, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 20;
    const offset = (page - 1) * limit;

    // Get total count
    const countResult = await executeQuery('SELECT COUNT(*) as total FROM shopify_orders');
    const totalOrders = countResult[0].total;
    const totalPages = Math.ceil(totalOrders / limit);

    // Get orders with pagination
    const query = `
      SELECT 
        id, shopify_order_id, order_number, email, customer_name, customer_phone,
        total_price, currency, financial_status, fulfillment_status, 
        created_at, shopify_created_at, tags, test
      FROM shopify_orders 
      ORDER BY created_at DESC 
      LIMIT ? OFFSET ?
    `;

    const results = await executeQuery(query, [limit, offset]);
    
    res.json({
      orders: results,
      pagination: {
        currentPage: page,
        totalPages,
        totalOrders,
        hasNext: page < totalPages,
        hasPrev: page > 1
      }
    });
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Get single Shopify order details (protected route)
router.get('/orders/:id', authenticate, async (req, res) => {
  try {
    const orderId = req.params.id;
    
    const results = await executeQuery('SELECT * FROM shopify_orders WHERE id = ? OR shopify_order_id = ?', [orderId, orderId]);
    
    if (results.length === 0) return res.status(404).json({ message: 'Order not found' });
    
    const order = results[0];
    // Parse JSON fields for better readability
    try {
      order.line_items = JSON.parse(order.line_items || '[]');
      order.shipping_address = JSON.parse(order.shipping_address || '{}');
      order.billing_address = JSON.parse(order.billing_address || '{}');
      order.raw_data = JSON.parse(order.raw_data || '{}');
    } catch (parseError) {
      console.error('JSON parse error:', parseError);
    }
    
    res.json(order);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Sync orders from Shopify API (manual sync - protected route)
router.post('/sync', authenticate, async (req, res) => {
  try {
    if (!process.env.SHOPIFY_STORE_URL || !process.env.SHOPIFY_ACCESS_TOKEN) {
      return res.status(400).json({ message: 'Shopify credentials not configured' });
    }

    const shopifyUrl = `https://${process.env.SHOPIFY_STORE_URL}/admin/api/2025-07/orders.json`;
    const headers = {
      'X-Shopify-Access-Token': process.env.SHOPIFY_ACCESS_TOKEN,
      'Content-Type': 'application/json'
    };

    // Get recent orders from Shopify
    const response = await axios.get(shopifyUrl + '?status=any&limit=50', { headers });
    const orders = response.data.orders;

    let syncedCount = 0;
    let errorCount = 0;

    // Process each order
    for (const order of orders) {
      try {
        const orderData = {
          shopify_order_id: order.id,
          order_number: order.name || order.order_number,
          email: order.email,
          customer_name: order.customer ? `${order.customer.first_name || ''} ${order.customer.last_name || ''}`.trim() : '',
          customer_phone: order.customer ? order.customer.phone : null,
          total_price: parseFloat(order.total_price),
          currency: order.currency,
          financial_status: order.financial_status,
          fulfillment_status: order.fulfillment_status,
          order_status_url: order.order_status_url,
          line_items: JSON.stringify(order.line_items),
          shipping_address: JSON.stringify(order.shipping_address),
          billing_address: JSON.stringify(order.billing_address),
          shopify_created_at: new Date(order.created_at),
          shopify_updated_at: new Date(order.updated_at),
          tags: order.tags,
          note: order.note,
          processed_at: order.processed_at ? new Date(order.processed_at) : null,
          gateway: order.gateway,
          test: order.test || false,
          raw_data: JSON.stringify(order)
        };

        const insertQuery = `
          INSERT INTO shopify_orders (
            shopify_order_id, order_number, email, customer_name, customer_phone,
            total_price, currency, financial_status, fulfillment_status, order_status_url,
            line_items, shipping_address, billing_address, shopify_created_at, shopify_updated_at,
            tags, note, processed_at, gateway, test, raw_data
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
          ON DUPLICATE KEY UPDATE
            order_number = VALUES(order_number),
            email = VALUES(email),
            customer_name = VALUES(customer_name),
            customer_phone = VALUES(customer_phone),
            total_price = VALUES(total_price),
            currency = VALUES(currency),
            financial_status = VALUES(financial_status),
            fulfillment_status = VALUES(fulfillment_status),
            order_status_url = VALUES(order_status_url),
            line_items = VALUES(line_items),
            shipping_address = VALUES(shipping_address),
            billing_address = VALUES(billing_address),
            shopify_updated_at = VALUES(shopify_updated_at),
            tags = VALUES(tags),
            note = VALUES(note),
            processed_at = VALUES(processed_at),
            gateway = VALUES(gateway),
            test = VALUES(test),
            raw_data = VALUES(raw_data),
            updated_at = CURRENT_TIMESTAMP
        `;

        const values = [
          orderData.shopify_order_id,
          orderData.order_number,
          orderData.email,
          orderData.customer_name,
          orderData.customer_phone,
          orderData.total_price,
          orderData.currency,
          orderData.financial_status,
          orderData.fulfillment_status,
          orderData.order_status_url,
          orderData.line_items,
          orderData.shipping_address,
          orderData.billing_address,
          orderData.shopify_created_at,
          orderData.shopify_updated_at,
          orderData.tags,
          orderData.note,
          orderData.processed_at,
          orderData.gateway,
          orderData.test,
          orderData.raw_data
        ];

        await executeQuery(insertQuery, values);
        syncedCount++;
      } catch (orderError) {
        console.error(`Error syncing order ${order.id}:`, orderError);
        errorCount++;
      }
    }

    res.json({
      message: 'Sync completed',
      synced: syncedCount,
      errors: errorCount,
      total: orders.length
    });

  } catch (error) {
    console.error('Sync error:', error);
    res.status(500).json({ message: 'Sync failed', error: error.message });
  }
});

// Get Shopify orders statistics (protected route)
router.get('/stats', authenticate, async (req, res) => {
  try {
    const statsQuery = `
      SELECT 
        COUNT(*) as total_orders,
        SUM(total_price) as total_revenue,
        AVG(total_price) as average_order_value,
        COUNT(CASE WHEN fulfillment_status = 'fulfilled' THEN 1 END) as fulfilled_orders,
        COUNT(CASE WHEN financial_status = 'paid' THEN 1 END) as paid_orders,
        COUNT(CASE WHEN test = 1 THEN 1 END) as test_orders
      FROM shopify_orders
    `;

    const results = await executeQuery(statsQuery);
    res.json(results[0]);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
